/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        $Id: pgpTrust.c,v 1.11 2001/01/25 22:11:13 jeffc Exp $
____________________________________________________________________________*/

#include <math.h>

#include "pgpConfig.h"

#include "pgpDebug.h"
#include "pgpEnv.h"
#include "pgpKeyPriv.h"
#include "pgpTrustPriv.h"

	PGPByte
pgpTrustToExtern(PGPUInt16 trust)
{
	if (trust == PGP_TRUST_INFINITE)
		return (PGPByte) PGP_NEWTRUST_INFINITE;
	trust >>= TRUST_CERTSHIFT;
	if (trust > PGP_NEWTRUST_MAX)
		trust = PGP_NEWTRUST_MAX;
	return (PGPByte) trust;
}

	PGPUInt16
pgpTrustToIntern(PGPByte trust)
{
	if (trust > PGP_NEWTRUST_MAX) {
		if (trust == PGP_NEWTRUST_INFINITE)
			return (PGPUInt16) PGP_TRUST_INFINITE;
		else
			return 0;	/* Undefined */
	}
	return (PGPUInt16) trust << TRUST_CERTSHIFT;
}

/* Get marginal trust (extern format) from environment */
	PGPByte
pgpenvMarginalTrust( PGPEnv const *env )
{
	PGPInt32 marginals;
	PGPInt32 thresh;
	PGPByte marginalTrust;

	thresh = pgpenvGetInt(env, PGPENV_TRUSTED, NULL, NULL);
	thresh = (thresh > PGP_NEWTRUST_INFINITE) ?
				PGP_NEWTRUST_INFINITE : (thresh < 0) ? 0 : thresh;
	marginals = pgpenvGetInt(env, PGPENV_MARGINALS, NULL, NULL);
	marginalTrust = (PGPByte) ( (marginals < 1) ? 0
								: (thresh+marginals-1)/marginals );
	return marginalTrust;
}

/* Get complete trust (extern format) from environment */
	PGPByte
pgpenvCompleteTrust( PGPEnv const *env )
{
	PGPInt32 completes;
	PGPInt32 thresh;
	PGPByte completeTrust;

	thresh = pgpenvGetInt(env, PGPENV_TRUSTED, NULL, NULL);
	thresh = (thresh > PGP_NEWTRUST_INFINITE) ?
				PGP_NEWTRUST_INFINITE : (thresh < 0) ? 0 : thresh;
	completes = pgpenvGetInt(env, PGPENV_COMPLETES, NULL, NULL);
	completeTrust = (PGPByte) ((completes < 1) ? 0
								: (thresh+completes-1)/completes );
	return completeTrust;
}


/* Convert from old 3-level trust to the new 256-level (extern format) */
	PGPByte
pgpTrustOldToExtern(PGPEnv const *env, PGPByte oldTrust)
{
	PGPByte externTrust;

	switch (oldTrust & kPGPKeyTrust_Mask) {
	  case kPGPKeyTrust_Ultimate:
		externTrust = PGP_NEWTRUST_INFINITE;
		break;
	  case kPGPKeyTrust_Complete:
		externTrust = pgpenvCompleteTrust( env );
		break;
	  case kPGPKeyTrust_Marginal:
		externTrust = pgpenvMarginalTrust( env );
		break;
	  case kPGPKeyTrust_Undefined:
		externTrust = PGP_NEWTRUST_UNDEFINED;
		break;
	  case kPGPKeyTrust_Never:
	  default:
		externTrust = 0;
		break;
	}
	return externTrust;
}

/* Convert from new 256-level (extern format) to old 3-level trust  */
/* Divide into intervals so a series of Complete trust signatures will
 * not decay to a Marginal after just two are put together. */
	PGPByte
pgpTrustExternToOld(PGPEnv const *env, PGPByte externTrust)
{
	PGPUInt32 completeconf;
	PGPUInt32 marginalconf;

	if (externTrust >= PGP_NEWTRUST_INFINITE)
		return kPGPKeyTrust_Ultimate;
	if (externTrust == PGP_NEWTRUST_UNDEFINED)
		return kPGPKeyTrust_Unknown;

	completeconf = pgpenvCompleteTrust( env );
	marginalconf = pgpenvMarginalTrust( env );
	if (externTrust >=
			(2*completeconf + marginalconf) / 3)
		return kPGPKeyTrust_Complete;
	if (externTrust >= (2 * marginalconf) / 3)
		return kPGPKeyTrust_Marginal;
	return kPGPKeyTrust_Never;
}


/* Convert from old-format 7 level key trust to 3 level userid trust */
	PGPByte
pgpTrustOldKeyToUserID(PGPByte keyTrust)
{
	switch( keyTrust & kPGPKeyTrust_Mask ) {
		case kPGPKeyTrust_Undefined:
		case kPGPKeyTrust_Unknown:
		default:
			return kPGPUserIDTrust_Unknown;
		case kPGPKeyTrust_Never:
			return kPGPUserIDTrust_Untrusted;
		case kPGPKeyTrust_Marginal:
			return kPGPUserIDTrust_Marginal;
		case kPGPKeyTrust_Complete:
		case kPGPKeyTrust_Ultimate:
			return kPGPUserIDTrust_Complete;
	}
}

static double k1, k2;
static double kmaxt;	/* Maximum value of trust in terms of 1/n error prob */

static void
pgpTrustInit(void)
{
	double d = log(10.0);

	k1 = d/PGP_TRUST_DECADE;
	k2 = PGP_TRUST_DECADE/d;
	kmaxt = exp (PGP_TRUST_MAX * k1);
}

#define PGPTRUSTINIT(k) if (!k) pgpTrustInit()

#if 0

unsigned
pgpIntToTrust(unsigned long r)
{
	PGPTRUSTINIT(k2);
	return (unsigned) (log ((double) r) * k2 + 0.5);
}

unsigned long
pgpTrustToInt(unsigned t)
{
	PGPTRUSTINIT(k1);
	return (unsigned long) (exp (t * k1) + 0.5);
}

#ifdef __GNUC__		/* XXX: Should be something like HAS_LONG_LONG */
unsigned long long
pgpTrustToUll(unsigned t)
{
	PGPTRUSTINIT(k1);
	return (unsigned long long) (exp (t * k1) + 0.5);
}
#endif


#endif

double
pgpTrustToDouble(PGPUInt16 t)
{
	PGPTRUSTINIT(k1);
	return exp (t * k1);
}

	PGPUInt16
pgpDoubleToTrust(double d)
{
	pgpAssert (d >= .9);
	PGPTRUSTINIT(k2);
	if (d >= kmaxt)
		return PGP_TRUST_MAX;
	else
		return (unsigned) (log (d) * k2 + 0.5);
}


/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
